package hello;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.lang.management.ManagementFactory;
import java.util.List;

import javax.management.InstanceAlreadyExistsException;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.EntityTransaction;
import javax.persistence.Persistence;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.ejb.EntityManagerImpl;
import org.hibernate.jmx.StatisticsService;

public class HelloWorld {

    private EntityManagerFactory emf;
    private EntityManager em;
    private EntityTransaction tx;

    public static void main(String[] args) {
        HelloWorld helloworld = new HelloWorld();
        helloworld.initialize();

        while (true) {
            try {
                String input = helloworld.getUserInput();
                if (input == null) {
                    break;
                }

                String[] params = input.split(" +");
                if (params[0].equals("help")) {
                    printHelp();
                }
                else if (params[0].equals("quit") || params[0].equals("exit"))
                {
                    break;
                }
                else if (params[0].equals("add") && params[1].equals("message") && params.length == 3)
                {
                    helloworld.persistMessage(params[2]);
                }
                else if (params[0].equals("add") && params[1].equals("person") && params.length == 4)
                {
                    helloworld.persistPerson(params[2], params[3]);
                }
                else if (params[0].equals("delete") && params[1].equals("message") && params.length == 3)
                {
                    helloworld.removeMessage(new Long(params[2]).longValue());
                }
                else if (params[0].equals("delete") && params[1].equals("person") && params.length == 3)
                {
                    helloworld.removePerson(new Long(params[2]).longValue());
                }
                else if (params[0].equals("get") && params[1].equals("messages") && params.length == 2)
                {
                    helloworld.getAllMessages();
                }
                else if (params[0].equals("get") && params[1].equals("people") && params.length == 2)
                {
                    helloworld.getAllPeople();
                }
                else if (params[0].equals("get") && params[1].equals("people") && params.length == 3)
                {
                    helloworld.searchPeople(params[2]);
                }
                else {
                    printHelp();
                }
            }
            catch (Exception e) {
                System.err.println("Error occurred: " + e);
            }
        }

        System.out.println("Goodbye.");
        helloworld.shutdown();
    }

    private static void printHelp() {
        System.out.println("Valid commands:");
        System.out.println("  add message <text>");
        System.out.println("  add person <firstname> <lastname>");
        System.out.println("  delete message <id>");
        System.out.println("  delete person <id>");
        System.out.println("  get messages");
        System.out.println("  get people");
        System.out.println("  get people <search-string>");
        System.out.println("  help");
        System.out.println("  quit");
    }

    public String getUserInput() {
        String inputString = "";
        PrintStream out = System.out;
        BufferedReader in = new BufferedReader(new InputStreamReader(System.in));

        while ((inputString != null) && (inputString.trim().length() == 0)) {
            out.print("> ");
            try {
                out.flush();
                inputString = in.readLine();
            } catch (Exception e) {
                inputString = null;
            }
        }

        return inputString;
    }

    public void persistPerson(String firstName, String lastName) {
        Person person = new Person(firstName, lastName);
        startTransaction();
        try {
            getEntityManager().persist(person);
            commitTransaction();
        }
        catch (Exception e) {
            System.err.println("Failed to persist person. Cause: " + e);
            rollbackTransaction();
        }
    }

    public void removePerson(long id) {
        startTransaction();
        try {
            Person doomed = getEntityManager().find(Person.class, id);
            if (doomed != null) {
                getEntityManager().remove(doomed);
                System.out.println("Removed person with ID " + id);
            }
            else {
                System.out.println("There is no person with ID " + id);                
            }
            commitTransaction();
        }
        catch (Exception e) {
            rollbackTransaction();
        }
    }

    public void getAllPeople() {
        startTransaction();
        try {
            List people = getEntityManager().createNamedQuery("Person.getAll").getResultList();
            System.out.println( people.size() + " people found:" );
            for (Object p : people) {
                Person loadedPerson = (Person) p;
                System.out.println(loadedPerson.getId() + ": "
                    + loadedPerson.getFirstName() + " "
                    + loadedPerson.getLastName());
            }
            commitTransaction();
        }
        catch (Exception e) {
            System.err.println("Failed to get people. Cause: " + e);
            rollbackTransaction();
        }
    }

    public void searchPeople(String search) {
        startTransaction();
        try {
            search = '%' + search + '%';
            List people = getEntityManager().createNamedQuery("Person.search").setParameter("search", search).getResultList();
            System.out.println( people.size() + " people found in search:" );
            for (Object p : people) {
                Person loadedPerson = (Person) p;
                System.out.println(loadedPerson.getId() + ": "
                    + loadedPerson.getFirstName() + " "
                    + loadedPerson.getLastName());
            }
            commitTransaction();
        }
        catch (Exception e) {
            System.err.println("Failed to search for people. Cause: " + e);
            rollbackTransaction();
        }
    }
    
    public void persistMessage(String text) {
        Message message = new Message(text);
        startTransaction();
        try {
            getEntityManager().persist(message);
            commitTransaction();
        }
        catch (Exception e) {
            System.err.println("Failed to persist message. Cause: " + e);
            rollbackTransaction();
        }
    }

    public void removeMessage(long id) {
        startTransaction();
        try {
            Message doomed = getEntityManager().find(Message.class, id);
            if (doomed != null) {
                getEntityManager().remove(doomed);
                System.out.println("Removed message with ID " + id);
            }
            else {
                System.out.println("There is no message with ID " + id);                
            }
            commitTransaction();
        }
        catch (Exception e) {
            rollbackTransaction();
        }
    }

    public void getAllMessages() {
        startTransaction();
        try {
            List messages = getEntityManager().createNamedQuery("Message.getAll").getResultList();
            System.out.println( messages.size() + " message(s) found:" );
            for (Object m : messages) {
                Message loadedMsg = (Message) m;
                System.out.println(loadedMsg.getId() + ": " + loadedMsg.getText());
            }
            commitTransaction();
        }
        catch (Exception e) {
            System.err.println("Failed to get messages. Cause: " + e);
            rollbackTransaction();
        }
    }

    public void initialize() {
        enableHibernateStatistics(getEntityManager());
    }

    public void shutdown() {
        try {
            commitTransaction();
            if (this.emf != null) {
                this.emf.close();
                this.emf = null;
            }
        }
        catch (Exception e) {
            System.err.println("Failed to shutdown. Cause: " + e);
        }
    }

    public void commitTransaction() {
        if (this.tx != null) {
            this.tx.commit();
            this.tx = null;
        }

        if (this.em != null) {
            this.em.close();
            this.em = null;
        }
    }

    public void rollbackTransaction() {
        if (this.tx != null) {
            this.tx.rollback();
            this.tx = null;
        }

        if (this.em != null) {
            this.em.close();
            this.em = null;
        }
    }

    public EntityTransaction startTransaction() {
        if (this.tx == null) {
            this.tx = getEntityManager().getTransaction();
            this.tx.begin();
        }
        return this.tx;
    }

    public EntityManager getEntityManager() {
        if (this.em == null) {
            this.em = getEntityManagerFactory().createEntityManager();
        }
        return this.em;
    }

    public EntityManagerFactory getEntityManagerFactory() {
        if (this.emf == null) {
            this.emf = Persistence.createEntityManagerFactory("helloworld");
        }
        return this.emf;
    }

    public void enableHibernateStatistics(EntityManager entityManager)
    {
       final String HIBERNATE_STATISTICS_MBEAN_OBJECTNAME = "Hibernate:application=helloworld,type=statistics";
       try
       {
          StatisticsService mBean = new StatisticsService();
          SessionFactory sessionFactory = getHibernateSession(entityManager).getSessionFactory();
          mBean.setSessionFactory(sessionFactory);
          ObjectName objectName = new ObjectName(HIBERNATE_STATISTICS_MBEAN_OBJECTNAME);
          MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
          mbs.registerMBean(mBean, objectName);
          sessionFactory.getStatistics().setStatisticsEnabled(true);
       }
       catch (InstanceAlreadyExistsException iaee)
       {
          System.err.println("Duplicate MBean registration ignored: " + HIBERNATE_STATISTICS_MBEAN_OBJECTNAME);
       }
       catch (Exception e)
       {
          System.err.println("Could not register Hibernate statistics MBean. Cause: " + e);
       }
       return;
    }

    private static Session getHibernateSession(EntityManager entityManager) {
       Session session;
       if (entityManager.getDelegate() instanceof EntityManagerImpl) {
           EntityManagerImpl entityManagerImpl = (EntityManagerImpl) entityManager.getDelegate();
           session = entityManagerImpl.getSession();
       } else {
           session = (Session) entityManager.getDelegate();
       }
       return session;
    }
}
