/*
 * JBoss, a division of Red Hat.
 * Copyright 2005-2008, Red Hat Middleware, LLC. All rights reserved.
 */

package org.rhq.domain.correlation;

import javax.persistence.DiscriminatorValue;
import javax.persistence.Enumerated;

import org.rhq.core.domain.measurement.Availability;
import org.rhq.core.domain.measurement.AvailabilityType;


/**
 * Normalization Rule for Availabilities
 *
 * @author Heiko W. Rupp
 */
@DiscriminatorValue("AVAIL")
public class AvailabilityNormalizationRule extends NormalizationRule<Availability>
{

   /** Resource we are operating on */
   int resourceId;
   
   @Enumerated
   ResultState stateForUp;
   
   @Enumerated
   ResultState stateForDown;
   
   /**
    * Create a new Rule
    * @param resourceId The resource this should apply to
    * @param name The name of the rule
    * @param stateForUp The {@link ResultState} if the resource is UP
    * @param stateForDown The {@link ResultState} if the resource reports as DOWN
    */
   public AvailabilityNormalizationRule(int resourceId, String name, ResultState stateForUp, ResultState stateForDown) {
      ruleName = name;
      this.resourceId = resourceId;
      this.stateForDown = stateForDown;
      this.stateForUp = stateForUp;
   }

   /**
    * Normalize the provided input
    * @param availability The value to check
    * @return The rule result for the passed availability
    */
   public ResultState normalize(Availability availability)
   {
      ResultState resultState;
      if (availability.getAvailabilityType() == AvailabilityType.UP) 
         resultState = stateForUp;
      else
         resultState = stateForDown;
      return resultState;
   }
   
   /**
    * Return the resource id this rule is for 
    * @return stored resourceId
    */
   public int getResourceId()
   {
      return resourceId;
   }

   /**
    * Returns the result state that should be used if {@link AvailabilityType} is UP
    * @return result state for {@link AvailabilityType}.UP
    */
   public ResultState getStateForUp()
   {
      return stateForUp;
   }
   
   /**
    * Returns the result state that should be used if {@link AvailabilityType} is DOWN
    * @return result state for {@link AvailabilityType}.DOWN
    */
   public ResultState getStateForDown()
   {
      return stateForDown;
   }
}
