/*
 * JBoss, a division of Red Hat.
 * Copyright 2005-2008, Red Hat Middleware, LLC. All rights reserved.
 */

package org.rhq.domain.correlation;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.SequenceGenerator;
import javax.persistence.Table;

/**
 * A correlation rule, that takes normalized data as input.
 * Correlation rules serve to put multiple input into relation.
 * So it it possible to say "return red if at least one of the input are
 * red or more then 2 inputy are yellow".
 * The conditions themselves are stored in {@link CorrelationItem}s that
 * are hanging on this class.
 *
 * @author Heiko W. Rupp
 */
@Table(name="RHQ_CORR_CRULE")
@SequenceGenerator(name="RHQ_CORR_RULE_GEN", sequenceName="RHQ_CORR_CRULE_SEQ")
public class CorrelationRule
{

   /** Source rules that feed into this correlation rule */
   Set<String> sourceRules = new HashSet<String>();
   
   
   @GeneratedValue(strategy=GenerationType.SEQUENCE, generator="RHQ_CORR_RULE_GEN")
   @Id
   int id;
   
   /** Name of the rule */
   String ruleName;

   /** The items that make up the rule */
   @OneToMany
   List<CorrelationItem> items = new ArrayList<CorrelationItem>();

   
   /**
    * Create a new correlation rule. One source is required and may not be this rule.
    * @param name Name of the rule
    * @param source The first (and required) input source
    */
   public CorrelationRule(String name, String source) {
      ruleName = name;
      if (source==null)
         throw new IllegalArgumentException("Soure may never be null");
      if (source.equals(name))
         throw new IllegalArgumentException("Souce may not be the rulename");
      sourceRules.add(source);
   }
   
   /**
    * Return the list of all source rules for this rule
    * @return list of souces
    */
   public Set<String> getSourceRules()
   {
      return sourceRules;
   }

   /**
    * Human readable form for of this rule
    * @return a string representation of this rule
    */
   public String toString()
   {
      return "CorrelationRule[name=" + ruleName + ", src=" +sourceRules + "]";
   }

   /**
    * Get the name of this rule
    * @return name of this rule
    */
   public String getRuleName()
   {
      return ruleName;
   }

   /**
    * Obtain the {@link CorrelationItem}s for this rule
    * @return the items that make up this rule
    */
   public List<CorrelationItem> getItems()
   {
      return items;
   }

   /**
    * Set a new list of {@link CorrelationItem} for this rule
    * @param items a new List of {@link CorrelationItem}
    */
   public void setItems(List<CorrelationItem> items)
   {
      this.items = items;
   }
   
   /**
    * Add a new {@link CorrelationItem} to the list of items for this rule
    * @param item an additional {@link CorrelationItem}
    */
   public void addItem(CorrelationItem item) {
      items.add(item);
   }
}
