/*
 * JBoss, a division of Red Hat.
 * Copyright 2005-2008, Red Hat Middleware, LLC. All rights reserved.
 */
package org.rhq.domain.correlation;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.DiscriminatorValue;
import javax.persistence.OneToMany;
import javax.persistence.Transient;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.rhq.core.domain.measurement.MeasurementData;
import org.rhq.core.domain.measurement.MeasurementDataNumeric;
import org.rhq.core.domain.measurement.MeasurementDataTrait;
import org.rhq.core.domain.measurement.MeasurementSchedule;

/**
 * Normalization rule for metrics (Numeric, traits). The rule does not store the
 * conditions in itsels, but uses a list of {@link MetricNormRuleItem}s for this.
 * MetricNormalizationRules are keyed by the ID of the {@link MeasurementSchedule}
 * they operate on.
 *
 * @author Heiko W. Rupp
 */
@DiscriminatorValue("METRIC")
public class MetricNormalizationRule extends NormalizationRule<MeasurementData>
{
   @Transient
   transient Log log = LogFactory.getLog(MetricNormalizationRule.class);

   /** Metric we are operation on */
   int scheduleId;
   
   @OneToMany
   List<MetricNormRuleItem> items = new ArrayList<MetricNormRuleItem>();

   /**
    * Create a new rule
    * @param scheduleid The id of the {@link MeasurementSchedule} this rule is for
    * @param name name of the rule
    */
   public MetricNormalizationRule(int scheduleid, String name) {
      scheduleId = scheduleid;
      ruleName = name;
   }
   
   /**
    * Get the id of the {@link MeasurementSchedule} we're working on
    * @return the schedule id
    */
   public int getScheduleId()
   {
      return scheduleId;
   }

   /**
    * Obtain the list of rule items
    * @return list of rule items
    */
   public List<MetricNormRuleItem> getItems()
   {
      return items;
   }

   /**
    * Set a list of rule items
    * @param items a List of {@link MetricNormRuleItem}
    */
   public void setItems(List<MetricNormRuleItem> items)
   {
      this.items = items;
   }

   /**
    * Add a new {@link MetricNormRuleItem} to this rule.
    * @param item
    */
   public void addItem(MetricNormRuleItem item)
   {
      items.add(item);
   }
   
   
   /**
    * Normalize the passed data into a {@link ResultState}
    * @param data {@link MeasurementData} -- currently only {@link MeasurementDataTrait} and {@link MeasurementDataNumeric}
    * are supported. Other subtypes will return {@link ResultState}.INVALID.
    * @return {@link ResultState} for a matching item or {@link ResultState}.INVALID otherwise.
    */
   public ResultState normalize(MeasurementData data) {
      
      ResultState resultState = ResultState.INVALID;
      
      if (data instanceof MeasurementDataNumeric) {
         resultState = evaluateNumeric((MeasurementDataNumeric) data);
      }
      else if (data instanceof MeasurementDataTrait) {
         resultState = evaluateTrait((MeasurementDataTrait) data);
      }
      else
         log.warn("Data type for " + data +" not yet supported, returning INVALID");
      
      return resultState;
   }

   
   private ResultState evaluateNumeric(MeasurementDataNumeric data) 
   {

      boolean found = false;
      ResultState resultState = ResultState.INVALID;
      Double dataValue = data.getValue();
      
      for (MetricNormRuleItem item : getItems()) {
         
         if (item.getOperator() == NormalizationOperator.IN_OPEN) {
            String[] vals = item.getValue().split(":");
            if (vals.length==2) {
               Double a = Double.valueOf(vals[0]);
               Double b = Double.valueOf(vals[1]);
               found = (a <= dataValue && dataValue <= b);
            }
         } else if (item.getOperator() == NormalizationOperator.IN_CLOSED) {
            String[] vals = item.getValue().split(":");
            if (vals.length==2) {
               Double a = Double.valueOf(vals[0]);
               Double b = Double.valueOf(vals[1]);
               found = (a < dataValue && dataValue < b);
            }
         } else if (item.getOperator() == NormalizationOperator.OUTSIDE) {
            String[] vals = item.getValue().split(":");
            if (vals.length==2) {
               Double a = Double.valueOf(vals[0]);
               Double b = Double.valueOf(vals[1]);
               found = (dataValue < a  || dataValue > b);
            }
            
         } else {
         
         
            Double toCompare = Double.valueOf(item.getValue());
            
            switch (item.getOperator()) {
               case LESS_THEN:
                  found = (dataValue < toCompare);
                  break;
               case LESS_EQUALS:
                  found = (dataValue <= toCompare);
                  break;
               case EQUALS:
                  found = (dataValue.equals(toCompare));  // compare with threshold ?
                  break;
               case GREATER_EQUALS:
                  found = (dataValue >= toCompare);
                  break;
               case GREATER_THEN:
                  found = (dataValue > toCompare);
                  break;
               case NOT_EQUALS:
                  found =(!dataValue.equals(toCompare));
                  break;
               case IS_NAN:
                  found = dataValue.isNaN();
                  break;
               default:
                     log.warn("Illegal comparator " + item.getOperator() + " for rule " + this);
            }
         }
         
         if (log.isTraceEnabled()) {
            log.trace("RuleItem " + item + " evalutates to " + resultState + " for schedule " + data.getScheduleId() );
         }
         if (found) {
            resultState = item.getResultState();
            break;
         }
      }
      return resultState;
   }
   
   private ResultState evaluateTrait(MeasurementDataTrait data) {
      
      boolean found = false;
      ResultState resultState = ResultState.INVALID;

      String dataValue = data.getValue();
      
      for (MetricNormRuleItem item : getItems()) {
         String toCompare = item.getValue();
         
         switch (item.getOperator()) {
            case CONTAINS:
               found = (dataValue.contains(toCompare));
               break;
            case NOT_CONTAINS:
               found = !(dataValue.contains(toCompare));
               break;
            case MATCHES:
               found = (dataValue.matches(toCompare));
               break;
            case NOT_MATCHES:
               found = !(dataValue.matches(toCompare));
               break;
            case STARTS_WITH:
               found = (dataValue.startsWith(toCompare));
               break;
            case ENDS_WITH:
               found = (dataValue.endsWith(toCompare));
               break;
            default:
               log.warn("Illegal comparator " + item.getOperator() + " for rule " + this);
         }
         if (log.isTraceEnabled()) {
            log.trace("RuleItem " + item + " evalutates to " + resultState + " for schedule " + data.getScheduleId() );
         }
         if (found) {
            resultState = item.getResultState();
            break;
         }
      }
      return resultState;
   }
   
}
