/*
 * JBoss, a division of Red Hat.
 * Copyright 2005-2008, Red Hat Middleware, LLC. All rights reserved.
 */

package org.rhq.enterprise.server.correlation;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.PostConstruct;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.CacheException;
import org.jboss.cache.Node;
import org.jboss.cache.TreeCache;
import org.rhq.core.domain.measurement.Availability;
import org.rhq.core.domain.measurement.MeasurementData;
import org.rhq.core.domain.operation.OperationDefinition;
import org.rhq.core.pluginapi.operation.OperationResult;
import org.rhq.domain.correlation.ActionRule;
import org.rhq.domain.correlation.AvailabilityNormalizationRule;
import org.rhq.domain.correlation.CorrelationItem;
import org.rhq.domain.correlation.MetricNormalizationRule;
import org.rhq.domain.correlation.CorrelationRule;
import org.rhq.domain.correlation.NormalizationRule;
import org.rhq.domain.correlation.NormalizedData;
import org.rhq.domain.correlation.OperationsNormalizationRule;
import org.rhq.domain.correlation.ResultState;
import org.rhq.enterprise.server.RHQConstants;

/**
 * Entry into the Correlation system. This class takes care of rule stores etc, as
 * well as evaluation of input data provided.
 *
 * @author Heiko W. Rupp
 */
@Stateless
public class CorrelationManagerBean
{
   private static final String CORRELATIONS = "/correlations/";

   private static final String METRIC_RULES = "/metricRules/";

   private static final String AVAILABILITY_RULES = "/availabilityRules/";

   private static final String OPERATION_RULES = "/operationRules/";

   private static final Log log = LogFactory.getLog(CorrelationManagerBean.class);
   
   /**
    * A cache instance for {@link NormalizedData}. This needs to be 
    * replicated upon data insertion.
    */
   private TreeCache workingMemory ;
   
   /**
    * A cache for the {@link CorrelationRule}s and {@link NormalizationRule}s.
    * The cache is partitioned as follows:
    * <ul>
    * <li>{@link CorrelationRule}s : /correlations/ ; the store is keyed by the name of the source 
    * rules that input into the correlation. Correlation data is stored in a List</li>
    * <li>NormalizationRules are keyed by the Integer id of the respecive reference data
    *   <ul>
    *   <li>{@link MetricNormalizationRule}s: /metricRules/ ; scheduleId</li>
    *   <li>{@link AvailabilityNormalizationRules}: /availabilityRules/ ; resourceId<li>
    *   <li>{@link OperationsNormalizationRule}s : /operationRules/ ; operationDefinitionId</li>
    *   </ul>
    * </ul>
    */
   private TreeCache rulesStore;
   
   @PersistenceContext(unitName = RHQConstants.PERSISTENCE_UNIT_NAME)
   private EntityManager entityManager;
   
   
   public CorrelationManagerBean()
   {
      // TODO remove for SLSB use
      init();
   }
   
   
   @PostConstruct
   private void init()
   {
      try
      {
         workingMemory = new TreeCache(); // TODO get from MBean
         workingMemory.startService();
         rulesStore = new TreeCache(); // TODO get from MBean
         rulesStore.startService();
      }
      catch (Exception e)
      {
         // TODO Auto-generated catch block
         e.printStackTrace();
      }
   }
   
   
   /**
    * Insert a new normalization rule for a metric into the system. 
    * @param rule a new {@link MetricNormalizationRule}
    */
   public void addMetricNormalizationRule(MetricNormalizationRule rule) {
      
      Integer id = Integer.valueOf(rule.getScheduleId());
      putIntoRulesStore(METRIC_RULES, id, rule);
   }
   
   /**
    * Insert a new normalization rule for Availability into the system.
    * @param rule a new {@link AvailabilityNormalizationRule}
    */
   public void addAvailabilityNormalizationRule(AvailabilityNormalizationRule rule) {
      
      Integer id = Integer.valueOf(rule.getResourceId());
      putIntoRulesStore(AVAILABILITY_RULES, id, rule);
   }
   
   /**
    * Insert a new normalization rule for {@link OperationResult} into the system
    * @param rule a new {@link OperationsNormalizationRule}
    */
   public void addOperationNormalizationRule(OperationsNormalizationRule rule) {
      
      Integer id = Integer.valueOf(rule.getOperationDefinitionId());
      putIntoRulesStore(OPERATION_RULES, id, rule);
   }
   
   
   /**
    * Inserts a new or updated correlation rule into the system.
    * @param rule a or updated {@link CorrelationRule}
    */
   public void addCorrelationRule(CorrelationRule rule) {
      
      // Get all sources for the passed rule and store a reference for each of them.
      for (String source : rule.getSourceRules()) {
         List<CorrelationRule> rules;
         if (!existsCorrelationRule(source)) {
            rules = new ArrayList<CorrelationRule>();
         }
         else
            rules = getCorrelationRules(source);
         
         rules.add(rule); // TODO replace existing rules!!
         putCorrelationRules(source, rules);
      }
   }
   
   /**
    * This method does the normalization work if there was a rule present for the schedule of the data.
    * If no {@link NormalizationRule} is present, it directly returns.
    * @param data {@link MeasurementData} 
    */
   public void insertMetrics(MeasurementData data) {
      
      Integer scheduleId = Integer.valueOf(data.getScheduleId());
      if (! existsInRulesStore(METRIC_RULES, scheduleId)) {
         if (log.isTraceEnabled())
            log.trace("No rule for schedule [" + scheduleId + "] found");
         return;
      }
      
      MetricNormalizationRule rule = (MetricNormalizationRule) getFromRulesStore(METRIC_RULES, scheduleId);
      ResultState resultState = ResultState.INVALID;
   
      resultState = rule.normalize(data);
      
      if (log.isDebugEnabled()) {
         log.debug("Data [" + data.getValue() + "] for schedule [" + data.getScheduleId() + "] got normalized as " + resultState);
      }
      
      // create normalized data item, insert it into working memory and fire evaluation
      insertAndFire(rule, resultState);
   }

   
   /**
    * Normalize inserted availabilities. This is different from above ({@link #insertMetrics(MeasurementData)}, 
    * as availabilities are for entire resources and not for individual metrics 
    * @param availability
    */
   public void insertAvailability(Availability availability) {
      
      Integer resourceId = Integer.valueOf(availability.getResource().getId());
      if (!existsInRulesStore(AVAILABILITY_RULES, resourceId)) {
         if (log.isTraceEnabled())
            log.trace("No rule for resource [" + resourceId + "] found");
         return;
      }
      
      AvailabilityNormalizationRule rule = (AvailabilityNormalizationRule) 
            getFromRulesStore(AVAILABILITY_RULES , resourceId);
      
      ResultState resultState = ResultState.INVALID;
      
      // evaluate
      resultState = rule.normalize(availability);
      
      if (log.isDebugEnabled()) {
         log.debug("Availability [" + availability.getAvailabilityType() + "] for resource [" + resourceId + "] got normalized as " + resultState);
      }
      
      // create normalized data item, insert into working memory and fire evaluation
      insertAndFire(rule, resultState);
   }
   
   /**
    * Normalize provided {@link OperationResult} for the id of an {@link OperationDefinition} 
    * @param definitionId Id of an {@link OperationDefinition} that matches the {@link OperationResult}
    * @param result the {@link OperationResult} to evaluate
    * @todo TODO this needs much more thinking and fixing of RHQ-976
    */
   public void insertOperationResult(int definitionId, OperationResult result) {

      if (!rulesStore.exists(OPERATION_RULES + definitionId, definitionId)) {
         if (log.isTraceEnabled())
            log.trace("No rule for OperationDefinition [" + definitionId + "] found");
         return;
      }
      
      OperationsNormalizationRule rule
         = (OperationsNormalizationRule) getFromRulesStore(OPERATION_RULES , definitionId);
      
      ResultState state = ResultState.INVALID;
      state = rule.normalize(result);
      
      // create normalized data item, insert into working memory and fire evaluation
      insertAndFire(rule, state);
   }
   
   /**
    * Creates a {@link NormalizedData} item, inserts into working memory and fires evaluation - this is
    * just a little helper.
    * @param rule The rule that created the data
    * @param resultState The computed state
    */
   private void insertAndFire(NormalizationRule rule, ResultState resultState) {
      
      String ruleName = rule.getRuleName();
      NormalizedData nd = new NormalizedData(ruleName,resultState);
      putIntoWorkingMemory(ruleName, nd);
      evaluateCorrelation(ruleName,nd);
   }
   
   
   
   /**
    * Workhorse for the correlation units. Get the registered rules for the passed sourceRuleName,
    * loop over them and evaluate them. Recursively call this method with freshly created {@link NormalizedData}
    * as result of evaluation.
    * @param sourceRuleName Name of the source rule that is sending data to us
    * @param data a data item.
    */
   private void evaluateCorrelation(String sourceRuleName, NormalizedData data)
   {
      if (log.isTraceEnabled())
         log.trace("Start evaluateCorrelation");
      
      if (!existsCorrelationRule(sourceRuleName)) {
         if (log.isDebugEnabled())
            log.debug("No correlation rule found for source [" + sourceRuleName + "]");
         return; // No rule found .. we're done  TODO: consider removing nd from working mem in this case
      }
      
      List<CorrelationRule> rules = getCorrelationRules(sourceRuleName);
      for ( CorrelationRule rule : rules ) {
         /*
          * Check if it is an ActionRule. If so we call its trigger
          * and we're done
          */
         if (rule instanceof ActionRule) {
            ActionRule actionRule = ((ActionRule)rule);
            actionRule.setState(data.getState());
            actionRule.trigger();
            continue;
         }
         
         /*
          * Otherwise, this is a real correlation rule.
          * First check if we have enough input, then evaluate and then fire
          * the next level.
          */
         
         // See if we have data for all inputs
         int count = 0;
         for (String source : rule.getSourceRules()) {
            if (workingMemoryContains(source))
               count++;
         }
         if (count < rule.getSourceRules().size()) {
            if (log.isDebugEnabled())
               log.debug("There is no data for all inputs of [" + rule + "]");
            continue;
         }
         
         // first count states for the data we have
         int red=0;
         int yellow=0;
         int green=0;
         int invalid=0;
         for (String source : rule.getSourceRules()) {
            NormalizedData nd = getFromWorkingMemory(source);
            switch (nd.getState()) {
               case RED: red++; break;
               case YELLOW: yellow++; break;
               case GREEN: green++; break;
               case INVALID: invalid++; break;
            }
         }
         if (log.isDebugEnabled())
            log.debug("Input was r=" + red + ", y=" + yellow + ", g=" + green + ", i=" + invalid);
         
         // Now compare the states with the correlation items for this rule
         ResultState resultState = ResultState.INVALID; 
         for (CorrelationItem item : rule.getItems()) {
            if (item.matches(red, yellow, green, invalid)) {
               resultState = item.getResultState();
               break;
            }
         }
      
         if (log.isDebugEnabled())
            log.debug("Rule [" + rule + "] evaluated to [" + resultState + "]" );
         
         // now insert our new facts into the working memory
         NormalizedData nd = new NormalizedData(rule.getRuleName(),resultState); 
         putIntoWorkingMemory(rule.getRuleName(), nd);
         
         // evaluate the freshly created data
         evaluateCorrelation(rule.getRuleName(), nd);
      }
   }
   
   
   @SuppressWarnings("unchecked")
   private NormalizedData getFromWorkingMemory(String ruleName)
   {
      NormalizedData ret = null;
      try
      {
         // rule name is fqn
         Node node = workingMemory.get(ruleName);
         ret = (NormalizedData) node.get(ruleName);
      }
      catch (CacheException e)
      {
         // TODO Auto-generated catch block
         e.printStackTrace();
      }
      return ret;
   }
   
   private void putIntoWorkingMemory(String ruleName, NormalizedData data)
   {
      try
      {
         // rule name is FQN
         workingMemory.put(ruleName, ruleName,data);
      }
      catch (CacheException e)
      {
         // TODO Auto-generated catch block
         e.printStackTrace();
      }
   }
   
   private boolean workingMemoryContains(String ruleName) 
   {
      // rule name is FQN
      return workingMemory.exists(ruleName, ruleName);
   }
   
   
   private void putIntoRulesStore(String store, Integer id, NormalizationRule rule)
   {
      try
      {
         rulesStore.put(store + id, id, rule);
      }
      catch (CacheException e)
      {
         // TODO Auto-generated catch block
         e.printStackTrace();
      }
   }
   
   private NormalizationRule getFromRulesStore(String store, Integer id)
   {
      NormalizationRule rule;
      try {
         rule = (NormalizationRule) rulesStore.get(store + id , id);
      }
      catch (CacheException e) {
         e.printStackTrace();
         rule = null;
      }
      return rule;
   }
   
   private boolean existsInRulesStore(String store, Integer id)
   {
      return rulesStore.exists(store + id);
   }
   
   @SuppressWarnings("unchecked")
   private List<CorrelationRule> getCorrelationRules(String source) {
      List<CorrelationRule> rules = new ArrayList<CorrelationRule>();
      try {
         rules = (List<CorrelationRule>) rulesStore.get(CORRELATIONS + source, source);
      }
      catch (CacheException e) {
         e.printStackTrace();
      }
      return rules;
   }
   
   private boolean existsCorrelationRule(String source) {
      return rulesStore.exists(CORRELATIONS + source);
   }
   
   private void putCorrelationRules(String sourceRule, List<CorrelationRule> rules) {
      try
      {
         rulesStore.put(CORRELATIONS + sourceRule, sourceRule, rules);
      }
      catch (CacheException e)
      {
         // TODO Auto-generated catch block
         e.printStackTrace();
      }
   }
   
   /**
    * Temporary for demonstration purposes
    * @return
    */
   @SuppressWarnings("unchecked")
   public List<ActionRule> getLatestActions()
   {
      
      Query q = entityManager.createQuery("SELECT a FROM ActionRule a ORDER BY a.ctime DESC LIMIT 10");
      List res = q.getResultList();
      return res;
   }
   
}
