/**
 * Copyright 2010 JBoss Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jbpm.process.workitem.rest;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.codec.binary.Base64;
import org.drools.core.process.instance.WorkItemHandler;
import org.kie.api.runtime.process.WorkItem;
import org.kie.api.runtime.process.WorkItemManager;

public class RESTWorkItemHandler implements WorkItemHandler {
	
	private String username;
	private String password;
	
	public RESTWorkItemHandler() {
	}
	
	public RESTWorkItemHandler(String username, String password) {
		this.username = username;
		this.password = password;
	}

    public void executeWorkItem(WorkItem workItem, WorkItemManager manager) {
    	try {
	    	// extract required parameters
	        String urlStr = (String) workItem.getParameter("Url");
	        String method = (String) workItem.getParameter("Method");
	        if (urlStr == null) {
	            throw new Exception("Url is a required parameter");
	        }
	        if (method == null || method.trim().length() == 0) {
	        	method = "GET";
	        }
	        Map<String,Object> params = workItem.getParameters();
	
	        // optional timeout config parameters, defaulted to 60 seconds
	        Integer connectTimeout = (Integer) params.get("ConnectTimeout");
	        if (connectTimeout==null) connectTimeout = 60000;
	        Integer readTimeout = (Integer) params.get("ReadTimeout");
	        if (readTimeout==null) readTimeout = 60000;
	
	        URL url = new URL(urlStr);
	        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
	        connection.setRequestMethod(method);
	        connection.setConnectTimeout(connectTimeout);
	        connection.setReadTimeout(readTimeout);
	        addAuthorization(connection, params);
	        connection.connect();
	        int responseCode = connection.getResponseCode();
	        Map<String, Object> results = new HashMap<String, Object>();
	        if (responseCode >= 200 && responseCode < 300) {
	            InputStreamReader r = new InputStreamReader((InputStream) connection.getContent());
	            StringWriter sw = new StringWriter();
	            char[] buffer = new char[1024 * 4];
	            int n = 0;
	            while (-1 != (n = r.read(buffer))) {
	                sw.write(buffer, 0, n);
	            }
	            postProcessResult(sw.toString(), results);
	        } else {
	            results.put("Status", responseCode);
	            results.put("StatusMsg", "endpoint " + urlStr + " could not be reached");
	        }
	        // notify manager that work item has been completed
	        manager.completeWorkItem(workItem.getId(), results);
    	} catch (Exception e) {
    		e.printStackTrace();
    		manager.abortWorkItem(workItem.getId());
    	}
    }
    
    protected void postProcessResult(String result, Map<String, Object> results) {
        results.put("Result", result);
    }

    protected void addAuthorization(HttpURLConnection connection, Map<String, Object> params) {
    	String u = (String) params.get("Username");
    	String p = (String) params.get("Password");
    	if (u == null || p == null) {
    		u = this.username;
    		p = this.password;
    	}
        if (u != null && p != null) {
            String b64Auth = Base64.encodeBase64String((u + ":" + p).getBytes());
            connection.setRequestProperty("Authorization", "Basic " + b64Auth);
        } 
    }

    public void abortWorkItem(WorkItem workItem, WorkItemManager manager) {
        // Do nothing, this work item cannot be aborted
    }

}
